# Cargar librerías necesarias
library(ggplot2)
library(factoextra)
library(cluster)
library(tidyr)
library(readxl)

# 1. Cargar los datos desde el archivo Excel
data <- read_excel("PCA prueba  - Lunes.xlsx")


# Verificar y limpiar los datos
data <- na.omit(data)  # Eliminar filas con valores faltantes
data <- scale(data)     # Estandarizar los datos (media=0, sd=1)

# 2. Calcular los componentes principales (PCA)
pca_result <- prcomp(data, scale = TRUE)

# 3. Panel de cargas (loadings) de los componentes principales
loadings <- as.data.frame(pca_result$rotation)
loadings$Variable <- rownames(loadings)
loadings <- tidyr::gather(loadings, key = "Componente", value = "Carga", -Variable)

ggplot(loadings, aes(x = Variable, y = Carga, fill = Carga)) +
  geom_bar(stat = "identity", position = "dodge", color = "black") +
  facet_wrap(~ Componente, scales = "free_y") +
  scale_fill_gradient2(low = "#FC4E07", mid = "white", high = "#00AFBB", midpoint = 0) +
  labs(title = "Cargas de los Componentes Principales para Datos de Propiedades",
       x = "Variables",
       y = "Carga",
       fill = "Carga") +
  theme_minimal() +
  theme(axis.text.x = element_text(angle = 45, hjust = 1))

# 4. Gráfico de las cargas (loadings) de los componentes principales
fviz_pca_var(pca_result, col.var = "contrib", 
             gradient.cols = c("#00AFBB", "#E7B800", "#FC4E07"),
             repel = TRUE) + 
  ggtitle("Cargas de los Componentes Principales - Datos de Propiedades") +
  theme_minimal()

# 5. Mostrar la varianza y la varianza acumulada
varianza <- pca_result$sdev^2
varianza_acumulada <- cumsum(varianza) / sum(varianza)

cat("Varianza explicada por cada componente:\n")
print(varianza)

cat("\nVarianza acumulada explicada:\n")
print(varianza_acumulada)

# 6. Gráfico de barras de la varianza de los componentes (Criterio del codo)
fviz_eig(pca_result, addlabels = TRUE, 
         barfill = "#2E9FDF", barcolor = "#2E9FDF") +
  ggtitle("Varianza Explicada por Componentes - Datos de Propiedades") +
  theme_minimal()

# 7. Gráfico de la varianza acumulada con umbral del 80%
varianza_df <- data.frame(
  Componente = 1:length(varianza_acumulada),
  Varianza_Acumulada = varianza_acumulada
)

ggplot(varianza_df, aes(x = Componente, y = Varianza_Acumulada)) +
  geom_line(color = "#2E9FDF", size = 1.2) +
  geom_point(color = "#FC4E07", size = 3) +
  geom_hline(yintercept = 0.8, color = "red", linetype = "dashed", size = 1) +
  scale_y_continuous(labels = scales::percent) +
  labs(title = "Varianza Acumulada de los Componentes - Datos de Propiedades",
       x = "Número de Componentes",
       y = "Varianza Acumulada") +
  theme_minimal()


